package tproxy

import (
	"net"
	"os"
	"strconv"
	"strings"
	"syscall"

	"github.com/txthinking/ant"
)

func ListenTCP(network string, laddr *net.TCPAddr) (*net.TCPListener, error) {
	l, err := net.ListenTCP(network, laddr)
	if err != nil {
		return nil, err
	}
	defer l.Close()

	f, err := l.File()
	if err != nil {
		return nil, err
	}
	defer f.Close()
	fd := int(f.Fd())
	if err := syscall.SetsockoptInt(fd, syscall.SOL_IP, syscall.IP_TRANSPARENT, 1); err != nil {
		return nil, err
	}
	tmp, err := net.FileListener(f)
	if err != nil {
		return nil, err
	}
	return tmp.(*net.TCPListener), nil
}

func DialTCP(network, addr string) (net.Conn, error) {
	taddr, err := net.ResolveTCPAddr("tcp", addr)
	if err != nil {
		return nil, err
	}
	saddr, err := tcpAddrToSocketAddr(taddr)
	if err != nil {
		return nil, err
	}
	fd, err := syscall.Socket(tcpAddrFamily("tcp", taddr, nil), syscall.SOCK_STREAM, syscall.IPPROTO_TCP)
	if err != nil {
		return nil, err
	}
	if err := syscall.SetsockoptInt(fd, syscall.SOL_SOCKET, syscall.SO_REUSEADDR, 1); err != nil {
		syscall.Close(fd)
		return nil, err
	}
	if err := syscall.SetsockoptInt(fd, syscall.SOL_IP, syscall.IP_TRANSPARENT, 1); err != nil {
		syscall.Close(fd)
		return nil, err
	}
	if err := syscall.SetNonblock(fd, true); err != nil {
		syscall.Close(fd)
		return nil, err
	}
	if err = syscall.Connect(fd, saddr); err != nil && !strings.Contains(err.Error(), "operation now in progress") {
		syscall.Close(fd)
		return nil, err
	}

	f := os.NewFile(uintptr(fd), string(ant.RandomNumber()))
	defer f.Close()

	c, err := net.FileConn(f)
	if err != nil {
		syscall.Close(fd)
		return nil, err
	}
	return c, nil
}

func tcpAddrToSocketAddr(addr *net.TCPAddr) (syscall.Sockaddr, error) {
	switch {
	case addr.IP.To4() != nil:
		ip := [4]byte{}
		copy(ip[:], addr.IP.To4())

		return &syscall.SockaddrInet4{Addr: ip, Port: addr.Port}, nil

	default:
		ip := [16]byte{}
		copy(ip[:], addr.IP.To16())

		zoneID, err := strconv.ParseUint(addr.Zone, 10, 32)
		if err != nil {
			return nil, err
		}

		return &syscall.SockaddrInet6{Addr: ip, Port: addr.Port, ZoneId: uint32(zoneID)}, nil
	}
}

func tcpAddrFamily(net string, laddr, raddr *net.TCPAddr) int {
	switch net[len(net)-1] {
	case '4':
		return syscall.AF_INET
	case '6':
		return syscall.AF_INET6
	}

	if (laddr == nil || laddr.IP.To4() != nil) &&
		(raddr == nil || raddr.IP.To4() != nil) {
		return syscall.AF_INET
	}
	return syscall.AF_INET6
}
